// Expected global variables:
/*global $ */

var consoleUtils = {

    // Constant for the amount of indented space in JSON 'stringified' objects
    jsonIndentCount: 12,

    // A set of types used in the console for different output items
    consoleNotifyType: {
        assert: "consoleItemError",
        error: "consoleItemError",
        info: "consoleItemInfo",
        log: "consoleItemLog",
        warn: "consoleItemWarn"
    },

    // A set of UI strings
    consoleUITypeStrings : {
        emptyArray: "[array] [ ]",
        emptyObject: "[object] { }"
    },
    
    // A set of filters
    consoleFilterId: {
        all: -1,
        error: 0,
        warning: 1,
        message: 2,
        log: 3
    },
    
    getDetailedTypeOf: function (value, constructors) {
        /// <summary>
        ///     Gets a string representing the type of the passed in value.
        ///     This supliments the built in typeof function by calculating the type of certain objects such as
        ///     array, date, and regex
        /// </summary>
        /// <param name="value" type="Object">
        ///     The javascript value to be checked for its real type
        /// </param>
        /// <param name="constructors" type="Object" optional="true">
        ///     An optional object that contains a set of constructors to check an "object" type against for further
        ///     sub typing into array, date, and regex
        ///     These objects need to come from the same window context as the value we are classifying
        /// </param>
        /// <returns type="String">
        ///     A string representing the type of this value, possible values include:
        ///     undefined, null, object, array, date, regex, string, number, function, and boolean
        /// </returns>
        /// <disable>JS3053.IncorrectNumberOfArguments,JS2005.UseShortFormInitializations</disable>
        if (value === undefined) {
            return "undefined";
        }

        var type = (typeof value);
        if (type === "object") {
            if (value) {
                if (typeof value.length === "number" && typeof value.propertyIsEnumerable === "function" && !(value.propertyIsEnumerable("length")) && typeof value.splice === "function") {
                    return "array";
                }

                // See if we have specific constructors to test against
                var arrayCon = (constructors && constructors.array ? constructors.array : (new Array()).constructor);
                var dateCon = (constructors && constructors.date ? constructors.date : (new Date()).constructor);
                var regexCon = (constructors && constructors.regex ? constructors.regex : (new RegExp()).constructor);

                try {
                    if (value.constructor === arrayCon) {
                        return "array";
                    } else if (value.constructor === dateCon) {
                        return "date";
                    } else if (value.constructor === regexCon) {
                        return "regex";
                    }
                } catch (e) {
                    // This object is not accessible
                }
            } else {
                return "null";
            }

            return "object";
        }

        return type;
    },

    isEmpty: function (obj) {
        /// <summary>
        ///     Checks if the passed in object is empty (such as { }) because it has no members
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The javascript object to be checked for members
        /// </param>
        /// <returns type="Boolean">
        ///     True if the object contains no members, False otherwise
        /// </returns>

        if (typeof obj === "object") {
            try {
                for (var i in obj) {
                    return false;
                }
            } catch (e) {
                // Cannot access this member therefore it must have one
                return false;
            }
        }

        return true;
    },

    wrapInQuotes: function (stringToWrap) {
        /// <summary>
        ///     Escapes a string so that it is wrapped in double quotes.
        /// </summary>
        /// <param name="stringToWrap" type="String">
        ///     The javascript string that is to be escaped and wrapped in quotes
        /// </param>
        /// <returns type="String">
        ///     The escaped string
        /// </returns>
        
        return "\"" + stringToWrap.replace(/\\"/g, "\"") + "\"";
    },
       
    highlightElementText: function (start, end) {
        /// <summary>
        ///     Highlights consecutive dom elements on the page as if the
        ///     user had selected the text with the mouse
        /// </summary>
        /// <param name="start" type="Object">
        ///     The first dom element to highlight
        /// </param>
        /// <param name="end" type="Object" optional="true">
        ///     Optional parameter that specifies the last dom element to highlight
        ///     If this parameter is undefined, then the function will only highlight the start element
        /// </param>
        
        var selection = window.getSelection();
        selection.removeAllRanges();

        // Highlight the text
        var range = document.createRange();
        if (end) {
            range.setStart(start, 0);
            range.setEnd(end, 1);
        } else {
            range.selectNode(start);
        }
        
        try {
            selection.addRange(range);
        } catch (ex) {
            // The addRange function can fail on hidden table elements, so fail gracefully.
        }
    },
    
    getVisibleHtmlElementText: function (element) {
        /// <summary>
        ///     Calculates the visible text value of a dom element and returns it
        /// </summary>
        /// <param name="element" type="Object">
        ///     The dom element to get the visible text for
        /// </param>
        /// <returns type="String">
        ///     All the visible text for this dom element
        /// </returns>
        
        // Ensure the element is visible
        if ($(element).is(":visible")) {
        
            // Text nodes can just return their value
            if (element.nodeType === 3) {
                return element.nodeValue;
            }
        
            // Otherwise we need to use recursion to get the text for all the children nodes
            var visibleText = "", i = 0;
            while (element.childNodes[i]) {
                visibleText += consoleUtils.getVisibleHtmlElementText(element.childNodes[i]);
                i++;
            }

            return visibleText;
        }
        
        return "";
    },
    
    createPadding: function (levels, singleLevelPadding) {
        /// <summary>
        ///     Generates a white space string that can be used for indenting
        /// </summary>
        /// <param name="levels" type="Number">
        ///     The number of indents to create
        /// </param>
        /// <param name="singleLevelPadding" type="String">
        ///     The padding string to use for one level
        /// </param>
        /// <returns type="String">
        ///     The white space string to use for padding indents
        /// </returns>

        // Check for no padding
        if (levels === 0) {
            return "";
        }
        
        // Add each level
        var padding = "";
        for (var i = 0; i < levels; i++) {
            padding += singleLevelPadding;
        }
        return padding;
    },
    
    getIndentedObjectString: function (obj, detailedType, stringPadding, indentString, newLineString, useEncodeHtml, useTrim) {
        /// <summary>
        ///     Generates a indented string representing an object
        /// </summary>
        /// <param name="obj" type="Object">
        ///     The javascript object to be turned into a pretty string
        /// </param>
        /// <param name="detailedType" type="String">
        ///     The detailedType string for this object (generated from getDetailedTypeOf function)
        /// </param>
        /// <param name="stringPadding" type="Number">
        ///     The amount of padding to add to each line
        /// </param>
        /// <param name="indentString" type="String">
        ///     The string used for padding for each level of indentation
        /// </param>
        /// <param name="newLineString" type="String">
        ///     The string that should be used to split up each line of text
        /// </param>
        /// <param name="useEncodeHtml" type="Boolean">
        ///     Should each line of text be encoded so that Html is displayed safely
        /// </param>
        /// <param name="useTrim" type="Boolean">
        ///     Should each line of text be trimmed to remove whitespace
        /// </param>
        /// <returns type="String">
        ///     The correctly indented string
        /// </returns>
        
        var text = "";
        var objectString = "" + obj;
        if ((/\S/).test(objectString)) {
            // Generate the padding for this string indent
            var indentCount = 0;
            
            var finalLines = [];
            if (detailedType === "string") {
                // Strings need to retain their spacing
                if (useEncodeHtml) { 
                    objectString = toolwindowHelpers.htmlEscape(objectString);
                } 
                text = objectString.replace(/(\r\n|\n\r|\r|\n)/g, newLineString);
            } else {
                // Split into lines, then process each one
                var lines = $.trim(objectString).split(/[\r\n]+/);
                for (var lineIndex = 0; lineIndex < lines.length; lineIndex++) {
                    if (lines[lineIndex] !== "") {
                        var indent = "";

                        // Get this line of text using the parameter selections
                        var lineText = lines[lineIndex];
                        if (useEncodeHtml) { 
                            lineText = toolwindowHelpers.htmlEscape(lineText);
                        } 
                        lineText = lineText.replace(/^\s+|\s+$/g, ""); // Trim white space
                        

                        if (detailedType === "function") {
                            if ((/^\}/).test(lineText)) {
                                indentCount--;
                            }
                            
                            for (var i = 0; i < indentCount; i++) {
                                indent += indentString;
                            }
                        
                            if ((/\{$/).test(lineText)) {
                                indentCount++;
                            }
                        }
                        
                        finalLines.push(indent + lineText);
                    }
                }

                var lineBreaks = newLineString;

                // Small functions should be condensed to a single line
                if (detailedType === "function" && finalLines.length === 3) {
                    finalLines[1] = $.trim(finalLines[1].replace(/^(&nbsp;)+/, ""));
                    finalLines[2] = $.trim(finalLines[2]);
                    lineBreaks = " ";
                }

                // Join up the lines into html with line breaks
                text = finalLines.join(lineBreaks);
            }
        }
        
        return text;
    }

};
// SIG // Begin signature block
// SIG // MIIalQYJKoZIhvcNAQcCoIIahjCCGoICAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFIYkigOxYpvs
// SIG // gmD2aDv1c2zGTv6qoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSIMIIEhAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTfT/fRsaE3QZvzGGexf3tI
// SIG // 68FjIDBKBgorBgEEAYI3AgEMMTwwOqAggB4AYwBvAG4A
// SIG // cwBvAGwAZQBVAHQAaQBsAHMALgBqAHOhFoAUaHR0cDov
// SIG // L21pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEA
// SIG // CFK6XVHOAkeCu6tAhGsyMunQjC/9LiRUksITDQHpKHTe
// SIG // Ra5icEadzC4Z4GCQt5HjHaF3sqClcy+3K2MDoWD1t2fr
// SIG // fqgL0cjU9Duvpm++MVUTsx68Cg3SG+nuchwGOI7xAVmp
// SIG // 1dX3lB0fr7l/+v/vsaRij3VWpeWanBSr01tRp23g+IyZ
// SIG // d22AW8vpUQhkYdJjWCiMFIzMMn6YSx83QmNwrUFO2FHv
// SIG // aP2f0eqT0MdypVwq8nm5e9BRSMFG+CqYfxBQsdch1q6I
// SIG // q4nbyIUsnuFNsxxHJ8r6mHPDEPuvmY2mvtk67vFE4JNO
// SIG // 6igDpydYCkCwWQymLgM+XtrUyWa/nr2aBqGCAh8wggIb
// SIG // BgkqhkiG9w0BCQYxggIMMIICCAIBATCBhTB3MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQg
// SIG // VGltZS1TdGFtcCBQQ0ECCmECkkoAAAAAACAwCQYFKw4D
// SIG // AhoFAKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEw
// SIG // HAYJKoZIhvcNAQkFMQ8XDTEzMDMxNTA2MzQwNFowIwYJ
// SIG // KoZIhvcNAQkEMRYEFFOIx8TJgXHAFp2l5pSwQpnZWMdz
// SIG // MA0GCSqGSIb3DQEBBQUABIIBAHtjLXLo1txAgK8bRVUh
// SIG // tJklWm5a8HacG9lYFmNyWk8lsVOfUL7HxXzTTalxg8ti
// SIG // obWhtxG1Qa2RB9Y021DL8PQ8H+TfW5JU2S0GZqPrqIvj
// SIG // pXTk09cp7HuBuqUCOWzJydok/1wN9xN9y+jDtSTS/1qs
// SIG // L8klIyV4J3No5RNg6aPbJCPUc5OeTTKRnCQvCrsKEGoG
// SIG // aG28bpn8hoeAF/4xGeaEJWQFGm4x9uAsnS9OtwcAvGUs
// SIG // UL1FZH01js7TuYCEzvj1xyjWezQwAvQapRLmhkAJcLNK
// SIG // 0xz5S95LatuuzLiGkLv6FXyuFwcQzUrt1NsdDZiqH3ZT
// SIG // EDanzif+y2owEtg=
// SIG // End signature block
